% CODE TO SIMULATE THE NUCLEOSSOME RECRUITMENT SYSTEM PRESENTED IN
% Dodd IB, Micheelsen MA, Sneppen K, Thon G. Theoretical analysis of epigenetic
% cell memory by nucleosome modification. Cell. 2007 May 18;129(4):813-22. 
% doi: 10.1016/j.cell.2007.02.053. PMID: 17512413.
%
% For the well-mixed system (neigh_size = N), the manifold is curved.
% However, as the neighbour size decreases, the manifold gets straighter
% and straighter.
% 
% Some interesting parameter choices: 
% Curved manifold: alpha = 0.55 / neigh_size = 60
% Straight manifold: alpha = 0.8 / neigh_size = 2
% Straight manifold: alpha = 0.9 / neigh_size = 1

clear variables

N = 60;
alpha = 0.8;
neigh_size = 2;
Tmax = 2000000;

% initialise states: 0 = Methylated, 2 = Acetylathed, 1 = Unmodified.
part_state = ones(N,1);
rec = zeros(3,Tmax);
rec_stat = zeros(N,ceil(Tmax/N));

for t=1:Tmax
    focus_particle = randi(60);

    % step 2A
    if rand <= alpha
        neigh_min = max(1,focus_particle-neigh_size);
        neigh_max = min(N,focus_particle+neigh_size);

        partner = randi([neigh_min neigh_max]);
        if part_state(partner) == 0
            part_state(focus_particle) = max(0,part_state(focus_particle)-1);
        elseif part_state(partner) == 2
            part_state(focus_particle) = min(2,part_state(focus_particle)+1);
        end

    % step 2B
    else
        u = rand;
        if u < 1/3
            part_state(focus_particle) = max(0,part_state(focus_particle)-1);
        elseif u <= 2/3
            part_state(focus_particle) = min(2,part_state(focus_particle)+1);
        end
    end
    
    % recording
    rec(1,t) = sum(part_state==0);
    rec(2,t) = sum(part_state==1);
    rec(3,t) = sum(part_state==2);

    if rem(t,60) == 0
        rec_stat(:,t/60) = part_state;
    end
end

% plot timeseries
f = figure('visible','on')
plot(1:Tmax, rec)
legend(["Methylated" "Unmodified" "Acethylated"])

% compute heatmap
data = rec([1 3],:)';
count = zeros(N+1,N+1);
for k=1:length(data(:,1))
    count(data(k,1)+1,data(k,2)+1) = count(data(k,1)+1,data(k,2)+1)+1;
end

% plot heatmap
figure
hHM = heatmap(count+count');
hHM.YDisplayData=flip(hHM.YDisplayData);
